@extends('layouts.admin')

@section('title', 'Manage Users')

@section('content')
<div class="container-fluid px-4">
    <h2 class="mt-4 mb-4 text-center text-primary fw-bold border-bottom pb-2"><i class="fas fa-users me-2"></i> Manage Users</h2>

    <!-- Search and Filter Section -->
    <div class="card shadow-sm mb-4 border-0 rounded-4 bg-gradient-light">
        <div class="card-body py-3">
            <form action="{{ route('admin.users') }}" method="GET" class="row g-3 align-items-center">
                <div class="col-lg-4 col-md-6">
                    <input type="text" name="search" class="form-control form-control-sm shadow-sm rounded-pill border-0 bg-white text-primary" placeholder="🔍 Search by Name or Email" value="{{ request('search') }}">
                </div>
                <div class="col-lg-3 col-md-6">
                    <select name="role" class="form-select form-select-sm shadow-sm rounded-pill border-0 bg-white text-primary">
                        <option value="">All Roles</option>
                        @foreach($roles as $role)
                            <option value="{{ $role->name }}" {{ request('role') === $role->name ? 'selected' : '' }}>{{ ucfirst($role->name) }}</option>
                        @endforeach
                    </select>
                </div>
                <div class="col-lg-2 col-md-6">
                    <button type="submit" class="btn btn-primary btn-sm shadow-sm rounded-pill w-100 text-white"><i class="fas fa-filter me-2"></i>Filter</button>
                </div>
                <div class="col-lg-3 col-md-6 text-end">
                    <a href="{{ route('admin.roles.index') }}" class="btn btn-info btn-sm shadow-sm rounded-pill text-white">
                        <i class="fas fa-tools me-2"></i> Manage Roles
                    </a>
                </div>
            </form>
        </div>
    </div>

    <!-- Users Table Section -->
    <div class="card shadow-sm border-0 rounded-4">
        <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center rounded-top border-bottom">
            <h5 class="mb-0 fw-bold"><i class="fas fa-list me-2"></i>User List</h5>
            <a href="{{ route('admin.users.create') }}" class="btn btn-success btn-sm shadow-sm rounded-pill text-white">
                <i class="fas fa-user-plus me-2"></i>Add New User
            </a>
        </div>
        <div class="card-body p-3">
            @if($users->isEmpty())
                <div class="alert alert-warning text-center fw-bold">
                    <i class="fas fa-exclamation-circle me-2"></i>No users found.
                </div>
            @else
                <div class="table-responsive">
                    <table class="table table-hover align-middle text-center table-borderless rounded shadow-sm">
                        <thead class="bg-secondary text-uppercase text-light border-bottom">
                            <tr>
                                <th scope="col" class="border-bottom border-light">#</th>
                                <th scope="col" class="border-bottom border-light">Name</th>
                                <th scope="col" class="border-bottom border-light">Email</th>
                                <th scope="col" class="border-bottom border-light">Role</th>
                                <th scope="col" class="border-bottom border-light">Status</th>
                                <th scope="col" class="border-bottom border-light">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white">
                            @foreach($users as $user)
                                <tr class="align-middle mb-3 rounded animate-hover border-bottom">
                                    <td class="fw-bold text-secondary">{{ $user->id }}</td>
                                    <td class="fw-bold text-dark text-start">
                                        <div class="d-flex align-items-center">
                                            <img src="{{ $user->profile_picture_url ?? 'https://via.placeholder.com/40' }}" alt="Profile Picture" class="rounded-circle me-3 border border-primary" width="30" height="30">
                                            <span class="text-truncate" style="max-width: 150px;">{{ $user->name }}</span>
                                        </div>
                                    </td>
                                    <td class="text-muted">{{ $user->email }}</td>
                                    <td>
                                        <span class="badge bg-info text-white rounded-pill px-2 py-1">{{ $user->roles->pluck('name')->join(', ') ?? 'No Role Assigned' }}</span>
                                    </td>
                                    <td>
                                        <!-- Toggle Active/Inactive Status -->
                                        <div class="form-check form-switch d-inline">
                                            <input type="checkbox" class="form-check-input toggle-status shadow-sm" data-user-id="{{ $user->id }}" id="isActiveToggle{{ $user->id }}" {{ $user->is_active ? 'checked' : '' }}>
                                            <label class="form-check-label ms-2 fw-bold" id="statusLabel{{ $user->id }}" for="isActiveToggle{{ $user->id }}">
                                                <span class="badge {{ $user->is_active ? 'bg-success' : 'bg-danger' }} text-white rounded-pill px-2 py-1">{{ $user->is_active ? 'Active' : 'Inactive' }}</span>
                                            </label>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="btn-group" role="group">
                                            <!-- Manage Role Button -->
                                            <button class="btn btn-info btn-sm shadow-sm rounded-pill manage-role-btn text-white" data-user-id="{{ $user->id }}" data-user-roles="{{ $user->roles->pluck('id')->join(',') }}" data-user-is-ib="{{ $user->is_ib }}" data-bs-toggle="modal" data-bs-target="#roleManagementModal" title="Manage Roles">
                                                <i class="fas fa-user-cog"></i>
                                            </button>

                                            <!-- Edit button -->
                                            <a href="{{ route('admin.users.edit', $user->id) }}" class="btn btn-warning btn-sm shadow-sm rounded-pill text-white" title="Edit User">
                                                <i class="fas fa-edit"></i>
                                            </a>

                                            <!-- Delete button -->
                                            <form action="{{ route('admin.users.delete', $user->id) }}" method="POST" class="d-inline" onsubmit="return confirm('Are you sure you want to delete this user?');">
                                                @csrf
                                                @method('DELETE')
                                                <button type="submit" class="btn btn-danger btn-sm shadow-sm rounded-pill text-white" title="Delete User">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <div class="d-flex justify-content-center mt-3">
                    {{ $users->links('pagination::bootstrap-5') }}
                </div>
            @endif
        </div>
    </div>
</div>

<!-- Role Management Modal -->
@if($roles->isNotEmpty())
    <div class="modal fade" id="roleManagementModal" tabindex="-1" aria-labelledby="roleManagementModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <form action="{{ route('admin.users.updateRole') }}" method="POST" id="updateRoleForm">
                @csrf
                <div class="modal-content rounded-4">
                    <div class="modal-header bg-primary text-white">
                        <h5 class="modal-title" id="roleManagementModalLabel"><i class="fas fa-user-cog me-2"></i>Manage User Roles</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body p-3">
                        <input type="hidden" name="user_id" id="modalUserId" value="">
                        <div class="form-group mb-3">
                            <label for="roles" class="fw-bold">Assign Roles</label>
                            <select name="roles[]" id="roles" class="form-select form-select-sm shadow-sm rounded-4 bg-white">
                                @foreach($roles as $role)
                                    <option value="{{ $role->id }}">{{ ucfirst($role->name) }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div class="form-check mt-2">
                            <input type="checkbox" name="is_ib" id="is_ib" class="form-check-input shadow-sm">
                            <label for="is_ib" class="form-check-label fw-bold">Is IB</label>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="submit" class="btn btn-primary shadow-sm rounded-pill text-white"><i class="fas fa-save me-2"></i>Save Changes</button>
                    </div>
                </div>
            </form>
        </div>
    </div>
@endif

<script>
    document.addEventListener('DOMContentLoaded', function () {
        // JavaScript to pass user data to the modal for role management
        document.querySelectorAll('.manage-role-btn').forEach(button => {
            button.addEventListener('click', function () {
                const userId = this.getAttribute('data-user-id');
                const userRoles = this.getAttribute('data-user-roles').split(',');
                const isIB = this.getAttribute('data-user-is-ib') === '1';

                document.getElementById('modalUserId').value = userId;
                document.getElementById('is_ib').checked = isIB;

                // Set roles in the modal
                const rolesSelect = document.getElementById('roles');
                for (let i = 0; i < rolesSelect.options.length; i++) {
                    rolesSelect.options[i].selected = userRoles.includes(rolesSelect.options[i].value);
                }
            });
        });

        // JavaScript for toggling the user status
        document.querySelectorAll('.toggle-status').forEach(toggle => {
            toggle.addEventListener('change', function () {
                const userId = this.getAttribute('data-user-id');
                const isActive = this.checked ? 1 : 0;

                // Update label text
                const statusLabel = document.getElementById('statusLabel' + userId);
                statusLabel.innerHTML = `<span class="badge ${isActive ? 'bg-success' : 'bg-danger'} text-white rounded-pill px-2 py-1">${isActive ? 'Active' : 'Inactive'}</span>`;

                // Send AJAX request to update user status
                fetch(`/admin/users/${userId}/toggle-status`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify({ is_active: isActive })
                }).then(response => {
                    if (!response.ok) {
                        alert('Failed to update user status.');
                    }
                }).catch(error => {
                    alert('An error occurred: ' + error);
                });
            });
        });
    });
</script>
@endsection