<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Casts\AsArrayObject;

class PaymentGatewaySetting extends Model
{
    // Defining fillable attributes for mass assignment
    protected $fillable = [
        'gateway_name',
        'is_active',
        'gateway_parameters',
        'success_url',
        'cancel_url',
        'environment',
        'description',
        'min_transaction_amount',
        'transaction_fee',
        'currency',
        'api_key',
        'gateway_type',
        'is_manual',
        'manual_payment_type',
        'api_secret',
        'webhook_url',
        'notification_email',
        'manual_payment_details'

    ];

    // Defining attributes to be cast into other types automatically
    protected $casts = [
        'is_active' => 'boolean',
        'gateway_parameters' => 'array',
        'transaction_fee' => 'float',
        'environment' => 'string',
    ];

    /**
     * Get the gateway's status as a human-readable label.
     *
     * @return string
     */
    public function getStatusLabelAttribute()
    {
        return $this->is_active ? 'Active' : 'Inactive';
    }

    /**
     * Get the formatted transaction fee.
     *
     * @return string
     */
    public function getFormattedTransactionFeeAttribute()
    {
        return number_format($this->transaction_fee, 2) . ' %';
    }

    /**
     * Define environment label.
     *
     * @return string
     */
    public function getEnvironmentLabelAttribute()
    {
        return ucfirst($this->environment);
    }

    /**
     * Scope a query to only include active gateways.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to filter by gateway name.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $gatewayName
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeByName($query, $gatewayName)
    {
        return $query->where('gateway_name', $gatewayName);
    }

    /**
     * Get parameters as an associative array
     *
     * @return array|null
     */
    public function getParametersAttribute()
    {
        return json_decode($this->gateway_parameters, true);
    }

    /**
     * Set parameters by passing an array
     *
     * @param array $value
     * @return void
     */
    public function setParametersAttribute($value)
    {
        $this->attributes['gateway_parameters'] = json_encode($value);
    }
}
