<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Auth;
use App\Models\User;
use App\Models\IBRelationship;
use App\Models\Referral;
use App\Models\IBCommissionLevel;
use Illuminate\Http\Request;

class UserIBController extends Controller
{
    /**
     * Show the IB dashboard for the logged-in user.
     *
     * @return \Illuminate\View\View
     */
    public function ibDashboard()
    {
        $user = Auth::user();

        // Direct referrals count
        $directReferralsCount = Referral::where('referrer_id', $user->id)->count();

        // Total referrals count including nested referrals
        $totalReferralsCount = IBRelationship::with('nestedReferrals')
            ->where('ib_id', $user->id)
            ->get()
            ->reduce(function ($carry, $referral) {
                return $carry + 1 + $referral->nestedReferrals->count();
            }, 0);

        // Total earnings
        $totalEarnings = Referral::where('referrer_id', $user->id)->sum('earnings');

        // Fetch direct referrals
        $referrals = IBRelationship::where('ib_id', $user->id)
            ->with('referredUser')
            ->get();

        return view('user.ib-dashboard', compact(
            'user',
            'directReferralsCount',
            'totalReferralsCount',
            'totalEarnings',
            'referrals'
        ));
    }

    /**
     * Calculate total earnings for the IB, including all nested referrals.
     *
     * @param int $ibId
     * @return float
     */
    private function calculateTotalEarnings($ibId): float
    {
        return IBRelationship::where('ib_id', $ibId)
            ->with('referredUser')
            ->get()
            ->reduce(function ($carry, $referral) {
                $earnings = $referral->referredUser->earnings ?? 0.0;
                return $carry + $earnings + $this->calculateTotalEarnings($referral->referred_user_id);
            }, 0.0);
    }

    /**
     * Display the IB tree for the logged-in user.
     *
     * @return \Illuminate\View\View
     */
    public function viewIBTree()
    {
        $user = Auth::user();

        // Build the IB tree recursively for the logged-in user
        $tree = $this->buildIBTree($user->id);

        return view('user.ib-tree', compact('user', 'tree'));
    }

    /**
     * Build the IB tree recursively while maintaining order and hierarchy.
     *
     * @param int $ibId
     * @return array
     */
    private function buildIBTree($ibId, $level = 1): array
    {
        $referrals = IBRelationship::where('ib_id', $ibId)
            ->with('referredUser')
            ->get();

        return $referrals->map(function ($referral) use ($level) {
            return [
                'user' => $referral->referredUser,
                'level' => $level, // Add the level to each node
                'children' => $this->buildIBTree($referral->referred_user_id, $level + 1),
            ];
        })->toArray();
    }



    /**
     * Fetch direct referrals for the logged-in user.
     *
     * @return \Illuminate\View\View
     */
    public function directReferrals()
    {
        $user = Auth::user();

        // Fetch direct referrals
        $directReferrals = IBRelationship::where('ib_id', $user->id)
            ->with('referredUser')
            ->get();

        return view('user.direct-referrals', compact('user', 'directReferrals'));
    }

    /**
     * Fetch all active referrals of the logged-in user.
     *
     * @return \Illuminate\View\View
     */
    public function activeReferrals()
    {
        $user = Auth::user();

        // Fetch active referrals recursively
        $activeReferrals = $this->getActiveReferrals($user->id);

        return view('user.active-referrals', compact('user', 'activeReferrals'));
    }

    /**
     * Retrieve all active referrals recursively.
     *
     * @param int $ibId
     * @return \Illuminate\Support\Collection
     */
    private function getActiveReferrals($ibId)
    {
        return IBRelationship::where('ib_id', $ibId)
            ->with('referredUser')
            ->get()
            ->reduce(function ($carry, $referral) {
                if ($referral->referredUser && $referral->referredUser->is_active) {
                    $carry->push($referral->referredUser);
                }
                return $carry->merge($this->getActiveReferrals($referral->referred_user_id));
            }, collect());
    }

    /**
     * Show the referral earnings of the logged-in user.
     *
     * @return \Illuminate\View\View
     */
    public function referralEarnings()
    {
        $user = Auth::user();

        // Calculate total earnings
        $totalEarnings = $this->calculateTotalEarnings($user->id);

        return view('user.referral-earnings', compact('user', 'totalEarnings'));
    }

    /**
     * Update the IB earnings based on commission levels.
     *
     * @param Request $request
     * @param int $ibId
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateEarnings(Request $request, $ibId)
    {
        $request->validate([
            'trade_amount' => 'required|numeric|min:0',
        ]);

        $ib = User::findOrFail($ibId);

        // Retrieve commission level for this IB's level
        $commissionLevel = IBCommissionLevel::where('level', $ib->level)->first();
        $commissionPercentage = $commissionLevel ? $commissionLevel->commission_percentage : 0;

        // Calculate earnings
        $earnings = $request->trade_amount * ($commissionPercentage / 100);

        // Create or update referral earnings
        Referral::updateOrCreate(
            ['referrer_id' => $ibId, 'referred_user_id' => $request->referred_user_id],
            ['earnings' => $earnings]
        );

        return redirect()->back()->with('success', 'Earnings updated successfully.');
    }
}
