<?php

namespace App\Http\Controllers;

use App\Models\Role;
use App\Models\Permission;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RoleController extends Controller
{
    /**
     * Display a listing of roles.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $roles = Role::with('permissions')->get();
        $roles = Role::with('permissions')->paginate(10); // Use pagination here
        return view('admin.roles.index', compact('roles'));
    }

    /**
     * Show the form for creating a new role.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $permissions = Permission::all();
        return view('admin.roles.create', compact('permissions'));
    }

    /**
     * Store a newly created role in the database.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|unique:roles,name|max:255',
            'permissions' => 'array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        DB::beginTransaction();

        try {
            $role = Role::create(['name' => $request->name]);

            if ($request->has('permissions')) {
                $role->permissions()->sync($request->permissions);
            }

            DB::commit();
            return redirect()->route('admin.roles.index')->with('success', 'Role created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('admin.roles.create')->with('error', 'Failed to create role: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for editing a role.
     *
     * @param int $id
     * @return \Illuminate\View\View
     */
    public function edit($id)
    {
        $role = Role::with('permissions')->findOrFail($id);
        $permissions = Permission::all();
        $assignedUsers = User::whereHas('roles', function ($query) use ($id) {
            $query->where('id', $id);
        })->get();

        return view('admin.roles.edit', compact('role', 'permissions', 'assignedUsers'));
    }

    /**
     * Update the specified role in the database.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|unique:roles,name,' . $id . '|max:255',
            'permissions' => 'array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        DB::beginTransaction();

        try {
            $role = Role::findOrFail($id);
            $role->update(['name' => $request->name]);

            if ($request->has('permissions')) {
                $role->permissions()->sync($request->permissions);
            }

            DB::commit();
            return redirect()->route('admin.roles.index')->with('success', 'Role updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('admin.roles.edit', $id)->with('error', 'Failed to update role: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified role from the database.
     *
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($id)
    {
        DB::beginTransaction();

        try {
            $role = Role::findOrFail($id);

            // Check if any user has this role
            if ($role->users()->count() > 0) {
                return redirect()->route('admin.roles.index')->with('error', 'Cannot delete role because it is assigned to users.');
            }

            $role->delete();

            DB::commit();
            return redirect()->route('admin.roles.index')->with('success', 'Role deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('admin.roles.index')->with('error', 'Failed to delete role: ' . $e->getMessage());
        }
    }

    /**
     * Assign users to a role.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function assignUsers(Request $request, $id)
    {
        $request->validate([
            'users' => 'array',
            'users.*' => 'exists:users,id',
        ]);

        DB::beginTransaction();

        try {
            $role = Role::findOrFail($id);

            if ($request->has('users')) {
                $users = User::whereIn('id', $request->users)->get();
                foreach ($users as $user) {
                    $user->roles()->syncWithoutDetaching([$role->id]);

                    // Update 'is_ib' based on whether the 'IB' role is assigned
                    if ($role->name === 'IB') {
                        $user->is_ib = true;
                    } else {
                        $user->is_ib = false;
                    }
                    $user->save();
                }
            }

            DB::commit();
            return redirect()->route('admin.roles.index')->with('success', 'Users assigned to role successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('admin.roles.index')->with('error', 'Failed to assign users: ' . $e->getMessage());
        }
    }

    public function removeUserRole($roleId, $userId)
    {
        $role = Role::findOrFail($roleId);
        $user = User::findOrFail($userId);

        // Detach the role from the user
        $user->roles()->detach($roleId);

        return redirect()->route('admin.roles.edit', $roleId)->with('success', 'Role removed from user successfully.');
    }


    public function updateRole(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'roles' => 'nullable|array',
            'roles.*' => 'exists:roles,id',
        ]);

        $user = User::findOrFail($request->user_id);
        $user->roles()->sync($request->roles ?? []);

        // Update is_ib field based on the checkbox input
        $user->is_ib = $request->has('is_ib');
        $user->save();

        return redirect()->route('admin.users')->with('success', 'User roles updated successfully.');
    }

    /**
     * Display all users assigned to a specific role.
     *
     * @param int $id
     * @return \Illuminate\View\View
     */
    public function showUsers($id)
    {
        $role = Role::with('permissions')->findOrFail($id);
        $users = User::whereHas('roles', function ($query) use ($id) {
            $query->where('id', $id);
        })->get();

        return view('admin.roles.show_users', compact('role', 'users'));
    }
}
