<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Referral;
use App\Models\User;
use App\Models\IBRelationship;
use App\Models\IBHierarchy;

class ReferralController extends Controller
{
    /**
     * Display the referrals for the authenticated user.
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $user = Auth::user();

        // Fetch referrals with pagination and calculate total earnings
        $referrals = Referral::where('referrer_id', $user->id)
            ->with('referredUser')
            ->paginate(10);

        $totalEarnings = Referral::where('referrer_id', $user->id)->sum('earnings');

        // Generate the referral link
        $referralLink = $this->generateReferralLink();

        // Generate referral growth data (last 6 months)
        $growthData = $this->calculateReferralGrowth($user->id);

        return view('user.referrals', compact('user', 'referrals', 'totalEarnings', 'referralLink', 'growthData'));
    }

    /**
     * Calculate referral growth data for the past 6 months.
     *
     * @param int $userId
     * @return array
     */
    private function calculateReferralGrowth($userId)
    {
        $growthData = [];
        $currentMonth = now();

        for ($i = 5; $i >= 0; $i--) {
            $month = $currentMonth->copy()->subMonths($i);
            $monthName = $month->format('F');
            $referralCount = Referral::where('referrer_id', $userId)
                ->whereMonth('created_at', $month->month)
                ->whereYear('created_at', $month->year)
                ->count();

            $growthData[] = [
                'month' => $monthName,
                'count' => $referralCount,
            ];
        }

        return $growthData;
    }
    /**
     * Show the referral details.
     *
     * @param int $id
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $referral = Referral::with(['referredUser', 'referrer'])->findOrFail($id);

        return view('user.referrals.show', compact('referral'));
    }

    /**
     * Display the referral statistics for the authenticated user.
     *
     * @return \Illuminate\View\View
     */
    public function statistics()
    {
        $user = Auth::user();

        // Total Referrals and Active Referrals Count
        $totalReferralsCount = $this->calculateTotalReferrals();
        $activeReferralsCount = Referral::where('referrer_id', $user->id)
            ->whereHas('referredUser', function ($query) {
                $query->where('is_active', true);
            })
            ->count();

        // Total Earnings
        $totalEarnings = Referral::where('referrer_id', $user->id)->sum('earnings');

        return view('user.referrals.statistics', compact('user', 'totalReferralsCount', 'activeReferralsCount', 'totalEarnings'));
    }

    /**
     * Generate a referral link for the authenticated user.
     *
     * @return string
     */
    public function generateReferralLink()
    {
        $user = Auth::user();
        return url('/register') . '?ref=' . $user->id;
    }

    /**
     * Track the status of a specific referral.
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function trackReferralStatus($id)
    {
        $referral = Referral::with('referredUser')->findOrFail($id);

        $status = $referral->referredUser && $referral->referredUser->is_active ? 'Active' : 'Inactive';

        return response()->json([
            'referral' => $referral,
            'status' => $status,
        ]);
    }

    /**
     * Display referrals filtered by status.
     *
     * @param string $status
     * @return \Illuminate\View\View
     */
    public function referralsByStatus($status)
    {
        $user = Auth::user();
        $statusBool = $status === 'active';

        // Fetch referrals based on status
        $referrals = Referral::where('referrer_id', $user->id)
            ->whereHas('referredUser', function ($query) use ($statusBool) {
                $query->where('is_active', $statusBool);
            })
            ->with('referredUser')
            ->paginate(10);

        return view('user.referrals.by-status', compact('user', 'referrals', 'status'));
    }

    /**
     * Calculate the total earnings by different types of referrals.
     *
     * @return \Illuminate\View\View
     */
    public function earningsByReferralType()
    {
        $user = Auth::user();

        // Direct Earnings
        $directEarnings = Referral::where('referrer_id', $user->id)
            ->whereHas('referredUser', function ($query) {
                $query->where('is_active', true);
            })
            ->sum('earnings');

        // Indirect Earnings (Earnings through referred users' referrals)
        $indirectEarnings = Referral::whereHas('referrer', function ($query) use ($user) {
            $query->where('referrer_id', $user->id);
        })->sum('earnings');

        return view('user.referrals.earnings', compact('user', 'directEarnings', 'indirectEarnings'));
    }

    /**
     * Utility method to calculate total referrals count for the authenticated user.
     *
     * @return int
     */
    public function calculateTotalReferrals()
    {
        $user = Auth::user();

        $directReferrals = Referral::where('referrer_id', $user->id)->get();
        $totalReferralsCount = $directReferrals->count();

        foreach ($directReferrals as $referral) {
            $nestedReferralsCount = Referral::where('referrer_id', $referral->referred_user_id)->count();
            $totalReferralsCount += $nestedReferralsCount;
        }

        return $totalReferralsCount;
    }

    /**
     * Display a hierarchy view of all referrals.
     *
     * @return \Illuminate\View\View
     */
    public function viewReferralHierarchy()
    {
        $user = Auth::user();

        // Fetch IB hierarchy
        $hierarchy = IBHierarchy::where('ib_id', $user->id)
            ->with('children.referredUser')
            ->get();

        return view('user.referrals.hierarchy', compact('user', 'hierarchy'));
    }
}
