<?php

namespace App\Http\Controllers;

use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\TransactionsExport;
use Illuminate\Support\Facades\DB;

class AdminTransactionController extends Controller
{
    /**
     * Display a list of transactions with advanced search, filter, and date range capabilities.
     *
     * @param Request $request
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $filters = [
            'user_id' => $request->input('user_id'),
            'type' => $request->input('type'),
            'status' => $request->input('status'),
            'start_date' => $request->input('start_date'),
            'end_date' => $request->input('end_date'),
            'search' => $request->input('search'),
        ];

        $query = Transaction::with('user');

        // Search functionality
        if ($filters['search']) {
            $query->where(function ($q) use ($filters) {
                $q->whereHas('user', function ($userQuery) use ($filters) {
                    $userQuery->where('name', 'like', '%' . $filters['search'] . '%')
                        ->orWhere('email', 'like', '%' . $filters['search'] . '%');
                })->orWhere('type', 'like', '%' . $filters['search'] . '%')
                    ->orWhere('status', 'like', '%' . $filters['search'] . '%');
            });
        }

        // Advanced filter functionality
        if ($filters['user_id']) {
            $query->where('user_id', $filters['user_id']);
        }

        if ($filters['type']) {
            $query->where('type', $filters['type']);
        }

        if ($filters['status']) {
            $query->where('status', $filters['status']);
        }

        // Date range filter
        if ($filters['start_date']) {
            $query->whereDate('created_at', '>=', $filters['start_date']);
        }

        if ($filters['end_date']) {
            $query->whereDate('created_at', '<=', $filters['end_date']);
        }

        // Fetch transactions with pagination
        $transactions = $query->latest()->paginate(20);

        // Fetch transaction counts for summary cards
        $pendingTransactionsCount = Transaction::where('status', 'pending')->count();
        $completedTransactionsCount = Transaction::where('status', 'completed')->count();
        $rejectedTransactionsCount = Transaction::where('status', 'rejected')->count();

        return view('admin.transactions.index', compact(
            'transactions',
            'filters',
            'pendingTransactionsCount',
            'completedTransactionsCount',
            'rejectedTransactionsCount'
        ));
    }

    /**
     * Approve a pending transaction and update the user's wallet balance.
     *
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function approve($id)
    {
        $transaction = Transaction::findOrFail($id);

        if ($transaction->status !== 'pending') {
            return redirect()->route('admin.transactions.index')->with('error', 'Transaction is not in pending state.');
        }

        DB::transaction(function () use ($transaction) {
            // Mark transaction as completed (since approved and completed are logically the same)
            $transaction->status = 'completed';
            $transaction->approved_by = Auth::id();
            $transaction->updated_status_at = now();
            $transaction->save();

            // Update the wallet balance if deposit or withdrawal is approved
            if ($transaction->type === 'deposit') {
                $transaction->user->wallet->increment('balance', $transaction->amount);
            } elseif ($transaction->type === 'withdrawal') {
                if ($transaction->user->wallet->balance < $transaction->amount) {
                    throw new \Exception('Insufficient wallet balance for withdrawal approval.');
                }
                $transaction->user->wallet->decrement('balance', $transaction->amount);
            }
        });

        return redirect()->route('admin.transactions.index')->with('success', 'Transaction approved successfully.');
    }

    /**
     * Reject a pending transaction.
     *
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function reject($id)
    {
        $transaction = Transaction::findOrFail($id);

        if ($transaction->status !== 'pending') {
            return redirect()->route('admin.transactions.index')->with('error', 'Transaction is not in pending state.');
        }

        // Mark transaction as rejected
        $transaction->status = 'rejected';
        $transaction->approved_by = Auth::id();
        $transaction->updated_status_at = now();
        $transaction->save();

        return redirect()->route('admin.transactions.index')->with('success', 'Transaction rejected successfully.');
    }

    /**
     * Export transactions to Excel.
     *
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function exportTransactions()
    {
        return Excel::download(new TransactionsExport, 'transactions.xlsx');
    }

    /**
     * View the details of a specific transaction.
     *
     * @param int $id
     * @return \Illuminate\View\View
     */
    public function view($id)
    {
        $transaction = Transaction::with('user')->findOrFail($id);
        return view('admin.transactions.view', compact('transaction'));
    }

    /**
     * Bulk approve transactions.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function bulkApprove(Request $request)
    {
        $request->validate([
            'transaction_ids' => 'required|array',
            'transaction_ids.*' => 'exists:transactions,id',
        ]);

        $transactionIds = $request->input('transaction_ids');

        DB::transaction(function () use ($transactionIds) {
            $transactions = Transaction::whereIn('id', $transactionIds)->where('status', 'pending')->get();

            foreach ($transactions as $transaction) {
                $transaction->status = 'completed';
                $transaction->approved_by = Auth::id();
                $transaction->updated_status_at = now();
                $transaction->save();

                if ($transaction->type === 'deposit') {
                    $transaction->user->wallet->increment('balance', $transaction->amount);
                } elseif ($transaction->type === 'withdrawal') {
                    if ($transaction->user->wallet->balance < $transaction->amount) {
                        throw new \Exception('Insufficient wallet balance for withdrawal approval.');
                    }
                    $transaction->user->wallet->decrement('balance', $transaction->amount);
                }
            }
        });

        return redirect()->route('admin.transactions.index')->with('success', 'Transactions approved successfully.');
    }

    /**
     * Bulk reject transactions.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function bulkReject(Request $request)
    {
        $request->validate([
            'transaction_ids' => 'required|array',
            'transaction_ids.*' => 'exists:transactions,id',
        ]);

        $transactionIds = $request->input('transaction_ids');

        DB::transaction(function () use ($transactionIds) {
            Transaction::whereIn('id', $transactionIds)->where('status', 'pending')
                ->update([
                    'status' => 'rejected',
                    'approved_by' => Auth::id(),
                    'updated_status_at' => now(),
                ]);
        });

        return redirect()->route('admin.transactions.index')->with('success', 'Transactions rejected successfully.');
    }
}
