<?php

namespace App\Http\Controllers;

use App\Models\SupportTicket;
use App\Models\SupportTicketReply;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class AdminSupportTicketController extends Controller
{
    /**
     * Display a listing of the tickets with advanced filtering, search, and sorting.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $query = SupportTicket::with('user');

        // Search functionality
        if ($request->has('search')) {
            $query->where('subject', 'like', '%' . $request->search . '%')
                ->orWhereHas('user', function ($q) use ($request) {
                    $q->where('name', 'like', '%' . $request->search . '%')
                        ->orWhere('email', 'like', '%' . $request->search . '%');
                });
        }

        // Filter by status
        if ($request->has('status') && in_array($request->status, ['open', 'in_progress', 'completed', 'closed'])) {
            $query->where('status', $request->status);
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $tickets = $query->paginate(10);

        return view('admin.support.index', compact('tickets'));
    }

    /**
     * Show the details of a specific ticket, its replies, and provide admin actions.
     *
     * @param int $id
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $ticket = SupportTicket::with(['user', 'replies.user'])->findOrFail($id);

        return view('admin.support.show', compact('ticket'));
    }

    // Handle admin reply to a ticket
    public function reply(Request $request, $id)
    {
        $request->validate([
            'message' => 'required|string',
            'attachment' => 'nullable|file|mimes:jpeg,png,jpg,pdf,doc,docx|max:2048',
        ]);

        $attachmentPath = null;
        if ($request->hasFile('attachment')) {
            $attachmentPath = $request->file('attachment')->store('ticket_replies', 'public');
        }

        SupportTicketReply::create([
            'ticket_id' => $id,
            'user_id' => null, // Admin reply
            'message' => $request->message,
            'attachment' => $attachmentPath,
        ]);

        return back()->with('success', 'Reply sent successfully.');
    }

    /**
     * Update the status of a ticket.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:open,in_progress,completed,closed',
        ]);

        $ticket = SupportTicket::findOrFail($id);
        $ticket->update(['status' => $request->status]);

        return redirect()->route('admin.support.index')->with('success', 'Ticket status updated successfully.');
    }


    /**
     * Delete a ticket and its replies.
     *
     * @param int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($id)
    {
        $ticket = SupportTicket::findOrFail($id);

        // Delete replies associated with the ticket
        $ticket->replies()->delete();

        // Delete the ticket
        $ticket->delete();

        return redirect()->route('admin.support.index')->with('success', 'Ticket deleted successfully.');
    }
}
